# Plate Recognizer Android SDK Integration Guide

## Overview
The Plate Recognizer Android SDK provides high-performance license plate recognition directly on Android devices. This SDK is delivered as a complete, ready-to-use Android project with optimized TensorFlow Lite models and native libraries for efficient processing on ARM64 devices.

## Table of Contents
- [Plate Recognizer Android SDK Integration Guide](#plate-recognizer-android-sdk-integration-guide)
  - [Overview](#overview)
  - [Table of Contents](#table-of-contents)
  - [Getting Started](#getting-started)
    - [1. Free Trial Setup](#1-free-trial-setup)
    - [2. SDK Contents](#2-sdk-contents)
  - [System Requirements](#system-requirements)
    - [Device Requirements](#device-requirements)
    - [Development Requirements](#development-requirements)
  - [Quick Start](#quick-start)
    - [Using the Provided Android Project](#using-the-provided-android-project)
    - [Building the APK](#building-the-apk)
  - [Integration Steps](#integration-steps)
    - [1. Project Setup](#1-project-setup)
    - [2. SDK Initialization](#2-sdk-initialization)
    - [3. Image Processing](#3-image-processing)
    - [4. Sample Response](#4-sample-response)
  - [API Reference](#api-reference)
    - [`Snapshot.enginePrepare()`](#snapshotengineprepare)
    - [`Snapshot.engineProc()`](#snapshotengineproc)
    - [`Snapshot.engineOnStop()`](#snapshotengineonstop)
  - [Project Structure](#project-structure)
  - [Optimization Guidelines](#optimization-guidelines)
    - [Threading](#threading)
    - [Image Processing](#image-processing)
    - [Memory Management](#memory-management)
  - [Error Handling \& Troubleshooting](#error-handling--troubleshooting)
    - [Built-in Validation](#built-in-validation)
    - [Error Handling](#error-handling)
    - [Common Error Codes](#common-error-codes)
  - [Best Practices](#best-practices)
    - [Implementation Guidelines](#implementation-guidelines)
    - [Security](#security)
    - [License Notes](#license-notes)
  - [FAQ](#faq)
  - [Support \& Resources](#support--resources)
    - [Documentation](#documentation)
    - [Technical Support](#technical-support)
  - [License Agreement](#license-agreement)

## Getting Started

### 1. Free Trial Setup
1. Sign up for a free trial at [platerecognizer.com](https://platerecognizer.com).
2. After signing up, you will receive an email with your credentials. You can also find them in your ParkPow account under the "Account" section.
   - **LICENSE_KEY**: Your Snapshot SDK license key. It starts with `platerec_`.
   - **TOKEN**: Your API authentication token.

### 2. SDK Contents
The SDK is delivered as a complete Android project containing:
- `app/` - The main application module.
  - `src/main/jniLibs/arm64-v8a/` - Native libraries for the ARM64 architecture.
    - `libPlateRec.so` - The core Plate Recognition library.
    - `libssl.so` & `libcrypto.so` - Dependencies for network communication and data encryption.
    - `libtensorflowlite.so` - The TensorFlow Lite library for running machine learning models.
  - `src/main/assets/` - Encrypted machine learning models.
    - `asset0.dat` - OCR and vehicle detection models.
    - `asset1.dat` - Make, model, and color (MMC) and direction detection models.
  - `src/main/java/` - Sample integration code in the `com.parkpow.platerec` package.
  - `build.gradle` - Pre-configured Gradle build script with all necessary dependencies and settings.
- `gradle/` - Gradle wrapper configuration.
- `build.gradle` - Project-level build configuration.
- `gradlew` / `gradlew.bat` - Gradle wrapper scripts for building the project from the command line.
- An APK file ready for installation in `app/build/outputs/apk/debug/`.

## System Requirements

### Device Requirements
- **Android Version**: 7.0 (API level 24) or higher.
- **Architecture**: ARM64 (arm64-v8a).
  - **Important**: The SDK only supports native ARM64 devices. Emulators for x86/x86_64 or devices using ARM translation layers (like Houdini) are **NOT** supported.
- **Memory (RAM)**:
  - Minimum: 2GB.
  - Recommended: 4GB for optimal performance, especially when processing high-resolution images or video streams.
- **Storage**:
  - ~100MB for the installed application.
  - Additional space is required for storing images.
- **Camera**: A camera is required for live recognition from a video stream.

### Development Requirements
- **Android Studio**: Ladybug (2024.2.1) or a newer stable version.
- **CMake**: 3.22+ (the provided project is already configured).
- **Android NDK**: 25+ (the provided project is already configured).
- **Kotlin**: 1.8+ (the project is written in Kotlin).

## Quick Start

### Using the Provided Android Project
1. Unzip the provided SDK package.
2. Open the `snapshot-android` directory in Android Studio.
3. In the `local.properties` file, add your `LICENSE_KEY` and `TOKEN`:
   ```properties
   LICENSE_KEY=your_snapshot_sdk_license
   TOKEN=your_platerec_token
   ```
4. Build and run the project on a connected ARM64 device. The sample app demonstrates how to initialize the SDK and process an image. See the included `MainActivity.kt` for a complete working example.

### Building the APK
You can build the APK from the command line using the Gradle wrapper:
```bash
# Navigate to the project directory
cd snapshot-android

# Build the debug APK
./gradlew assembleDebug

# Build the release APK
./gradlew assembleRelease

# The APK will be located in app/build/outputs/apk/debug/ or app/build/outputs/apk/release/
```

## Integration Steps

### 1. Project Setup
If you are integrating the SDK into your existing application:

1.  **Copy Native Libraries**: Copy the `.so` files from the provided project's `app/src/main/jniLibs/arm64-v8a/` directory to the same directory in your application.
2.  **Copy Model Assets**: Copy the `.dat` files from the provided project's `app/src/main/assets/` directory to the same directory in your application.
3.  **Add Permissions**: Add the following permissions to your `AndroidManifest.xml`:
    ```xml
    <uses-permission android:name="android.permission.INTERNET" />
    <uses-permission android:name="android.permission.CAMERA" />
    ```
    **Note**: For devices running Android 6.0 (API 23) or higher, request these permissions at runtime using `ActivityCompat.requestPermissions()` before initializing the SDK to avoid crashes.4.  **Configure `build.gradle`**: Ensure your app's `build.gradle` file is configured for ARM64 and a minimum SDK version of 24:
    ```gradle
    android {
        defaultConfig {
            minSdkVersion 24
            ndk {
                abiFilters 'arm64-v8a'
            }
        }
    }
    ```

### 2. SDK Initialization
Initialize the SDK once when your application starts. A good place for this is in your `Application` class.

```kotlin
import com.parkpow.platerec.Snapshot

class YourApplication : Application() {
    override fun onCreate() {
        super.onCreate()

        val success = Snapshot.enginePrepare(
            context = applicationContext,
            token = "YOUR_TOKEN",
            licenseKey = "YOUR_LICENSE_KEY",
            writablePath = applicationContext.filesDir.absolutePath,
            assetManager = applicationContext.assets
        )

        if (!success) {
            Log.e("PlateRecognizer", "SDK initialization failed")
        }
    }
}
```

### 3. Image Processing
To process an image, call `Snapshot.engineProc()`.

```kotlin
import com.parkpow.platerec.Snapshot

// Process an image from a file path
val resultJson = Snapshot.engineProc(
    filename = "/path/to/your/image.jpg",
    configJson = "{}", // Optional JSON configuration (use "{}" for defaults; see sample app for advanced options like {"threshold_d": 0.1, "threshold_o": 0.3, ...})
    regions = arrayOf("us", "eu"), // Specify regions
    doMmc = true, // Enable Make, Model, Color detection
    doDirection = true, // Enable direction detection
    cameraId = "front-camera", // Optional
    timestamp = System.currentTimeMillis().toString() // Optional
)

// Parse the JSON response
val jsonObject = JSONObject(resultJson)
```

### 4. Sample Response
A successful call to `engineProc()` returns a JSON string with the recognition results.

```json
{
    "processing_time": 0.123,
    "results": [
        {
            "box": {"xmin": 316, "ymin": 187, "xmax": 466, "ymax": 227},
            "plate": "ABC123",
            "score": 0.901,
            "candidates": [
                {"score": 0.901, "plate": "ABC123"}
            ]
        }
    ],
    "usage": {
        "calls": 1,
        "max_calls": 1000
    }
}
```

## API Reference

The SDK functionality is exposed through the `Snapshot` object.

### `Snapshot.enginePrepare()`
Initializes the SDK engine. This method must be called once before any processing can be done.

**Signature**:
```kotlin
fun enginePrepare(
    context: Context,
    token: String,
    licenseKey: String,
    writablePath: String,
    assetManager: AssetManager
): Boolean
```

**Parameters**:
- `context`: The application context.
- `token`: Your ParkPow API token.
- `licenseKey`: Your Snapshot SDK license key.
- `writablePath`: A path to a writable directory for the SDK to store cache files. `context.filesDir.absolutePath` is a good choice.
- `assetManager`: The application's `AssetManager` to load the models.

**Returns**: `true` if initialization was successful, `false` otherwise.

### `Snapshot.engineProc()`
Processes a single image file for license plate recognition.

**Signature**:
```kotlin
fun engineProc(
    filename: String,
    configJson: String,
    regions: Array<String>,
    doMmc: Boolean,
    doDirection: Boolean,
    cameraId: String,
    timestamp: String
): String
```

**Parameters**:
- `filename`: The absolute path to the image file.
- `configJson`: A JSON string for advanced configuration (e.g., `{"threshold": 0.5}`). `"{}"` for default.
- `regions`: An array of country codes to look for (e.g., `arrayOf("us", "br")`).
- `doMmc`: If `true`, enables Make, Model, and Color detection.
- `doDirection`: If `true`, enables vehicle direction detection.
- `cameraId`: An optional identifier for the camera that captured the image.
- `timestamp`: An optional timestamp for when the image was captured.

**Returns**: A JSON string containing the recognition results.

### `Snapshot.engineOnStop()`
Saves the SDK's state. Call this when your application is shutting down to ensure all data is saved correctly. A good place to call this is in your main `Activity`'s `onDestroy()` method.

**Signature**:
```kotlin
fun engineOnStop(context: Context)
```

**Parameters**:
- `context`: The application context.

## Project Structure

The provided Android project is structured as follows:
```
snapshot-android/
├── app/
│   ├── src/main/
│   │   ├── jniLibs/arm64-v8a/          # Pre-compiled native libraries
│   │   │   ├── libPlateRec.so          # Main recognition library
│   │   │   ├── libssl.so               # OpenSSL library
│   │   │   ├── libcrypto.so            # OpenSSL crypto library
│   │   │   └── libtensorflowlite.so    # TensorFlow Lite library
│   │   ├── assets/                     # Pre-encrypted model files
│   │   │   ├── asset0.dat              # OCR and detection models
│   │   │   └── asset1.dat              # MMC and direction models
│   │   ├── java/                       # Sample integration code in the `com.parkpow.platerec` package.
│   │   └── res/                        # Android resources
│   ├── build.gradle                    # Pre-configured app build script
│   └── build/outputs/apk/              # Built APK files (after building)
├── gradle/                             # Gradle wrapper
├── build.gradle                        # Project build configuration
├── gradlew                             # Gradle wrapper (Unix)
├── gradlew.bat                         # Gradle wrapper (Windows)
└── README.md                           # This file
```

## Optimization Guidelines

### Threading
- `enginePrepare()` and `engineOnStop()` **must** be called on the main thread.
- `engineProc()` is thread-safe and should be called on a background thread to avoid blocking the UI.

### Image Processing
- **Resolution**: For best results, use images with a resolution between 800x600 and 1920x1080.
- **Format**: The SDK supports JPEG and PNG formats.
- **Image Quality**:
  - Ensure the license plate is in focus and clearly readable.
  - Good, consistent lighting is crucial.
  - Minimize motion blur.
  - Avoid extreme angles.

### Memory Management
- The SDK is optimized for memory usage, but it's good practice to:
  - Process one image at a time per thread.
  - Downsample very large images before processing.
  - Call `engineOnStop()` in your main `Activity`'s `onDestroy()` method to clean up resources.

## Error Handling & Troubleshooting

### Built-in Validation
The SDK performs several checks at initialization:
- Device compatibility (ARM64).
- Available storage and RAM.
- License key validity.
- Network connectivity for the initial license check.

### Error Handling
Wrap calls to `engineProc()` in a `try-catch` block to handle potential exceptions.

```kotlin
try {
    val result = Snapshot.engineProc(...)
    // Process result
} catch (e: Exception) {
    Log.e("PlateRecognizer", "Recognition failed", e)
}
```

Additionally, check the JSON response from `engineProc()` for error fields (e.g., `{"error": "ERR0001"}`). Refer to Common Error Codes below.

### Common Error Codes
- `ERR0001`: Invalid or expired license key.
- `ERR0002`: Network error during the initial license check.
- `ERR0003`: Invalid image format or size.
- `ERR0016`: Device fingerprint does not match the license.
- `ERR0020`: Insufficient RAM or storage.
- `ERR0030`: Failed to load the machine learning models from assets.
- `ERR0040`: Camera initialization error.

## Best Practices

### Implementation Guidelines
1.  **Initialize Once**: Call `enginePrepare()` once at application startup.
2.  **Background Processing**: Always run `engineProc()` on a background thread.
3.  **Error Handling**: Implement robust error handling for all SDK calls.
4.  **Resource Management**: Call `engineOnStop()` when the app is destroyed (e.g., in your main `Activity`'s `onDestroy()`).
5.  **Image Quality**: Pre-process camera frames to ensure good image quality.

### Security
- The SDK uses encrypted models and license key validation to protect the intellectual property.
- Emulator detection and device fingerprinting prevent unauthorized usage.

### License Notes
- Each license key is tied to a specific device.
- An internet connection is required for the first-time license validation.
- The free trial comes with a limited number of recognitions.

## FAQ
**Q: Can I run the SDK on an emulator?**
A: No, the SDK requires a native ARM64 device and will not work on x86/x86_64 emulators.

**Q: Do I need an internet connection to use the SDK?**
A: An internet connection is required only for the initial license validation. After that, the SDK works completely offline.

**Q: What is the processing time per image?**
A: The processing time depends on the device and image resolution, but it typically ranges from 100ms to 300ms on modern devices.

**Q: Can I process a video stream?**
A: Yes, you can process frames from a video stream. Ensure you are doing the processing on a background thread to avoid frame drops.

**Q: How do I handle offline mode after initial setup?**
A: The SDK works fully offline after the initial license validation. Ensure network access only for the first run to validate the license key.

## Support & Resources

### Documentation
- Main Documentation: [guides.platerecognizer.com](https://guides.platerecognizer.com)
- API Reference: See the [API Reference](#api-reference) section in this document.

### Technical Support
- **Contact Us**: [platerecognizer.com/contact](https://platerecognizer.com/contact)
- **Email**: support@platerecognizer.com

## License Agreement
Usage of the Plate Recognizer SDK is subject to the terms of your license agreement. For more details, see [platerecognizer.com](https://platerecognizer.com).
