package com.parkpow.platerec

import android.os.Bundle
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.Card
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.parkpow.platerec.ui.theme.platerectestTheme
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import java.util.Locale
import kotlin.system.measureTimeMillis

private const val TAG = "PlateRec-M"

// Data models for structured results
private data class FileResult(val fileName: String, val result: String)

private data class RecognitionInfo(val summary: String, val results: List<FileResult>)

// Sealed interface for the outcome of the recognition process
private sealed interface RecognitionOutcome {
    data class Success(val info: RecognitionInfo) : RecognitionOutcome

    data class Failure(val message: String) : RecognitionOutcome
}

class MainActivity : ComponentActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()

        setContent {
            platerectestTheme {
                plateRecScreen(
                    runRecognition = { onStatus ->
                        // Run everything heavy off the main thread
                        withContext(Dispatchers.IO) { prepareAndRecognize(onStatus) }
                    },
                )
            }
        }
    }

    override fun onStop() {
        super.onStop()
        // Tell the SDK to save everything NOW.
        // We might never come back from this state.
        Snapshot.engineOnStop(this)
    }

    /**
     * Heavy path: prepares engine, scans assets folder for images, and runs recognition on all of them.
     * This MUST run on a background dispatcher (Dispatchers.IO).
     */
    private suspend fun prepareAndRecognize(onStatus: suspend (String) -> Unit): RecognitionOutcome {
        withContext(Dispatchers.Main) { onStatus("Preparing engine…") }

        // 1. Prepare Engine (Done once)
        val prepareMs =
            measureTimeMillis {
                val prepared =
                    Snapshot.enginePrepare(
                        this,
                        BuildConfig.TOKEN,
                        BuildConfig.LICENSE_KEY,
                        filesDir.absolutePath,
                        assets,
                    )
                if (!prepared) {
                    Log.e(TAG, "enginePrepare() failed")
                    return RecognitionOutcome.Failure("Prepare Failed")
                }
            }
        Log.i(TAG, "enginePrepare() took ${prepareMs}ms")

        // 2. Identify Image Files in Assets
        withContext(Dispatchers.Main) { onStatus("Scanning assets…") }

        val validExtensions = setOf("jpg", "jpeg", "png", "bmp", "webp")
        val imageFiles =
            try {
                assets.list("")?.filter { fileName ->
                    val ext = fileName.substringAfterLast('.', "").lowercase(Locale.ROOT)
                    validExtensions.contains(ext)
                } ?: emptyList()
            } catch (e: IOException) {
                Log.e(TAG, "Failed to list assets", e)
                return RecognitionOutcome.Failure("Error listing assets: ${e.message}")
            }

        if (imageFiles.isEmpty()) {
            return RecognitionOutcome.Success(RecognitionInfo("No images found in assets folder.", emptyList()))
        }

        // 3. Configuration Setup
        val configJson =
            """
            {
              "threshold_d": 0.1,
              "threshold_o": 0.3,
              "mode": "",
              "detection_rule": "strict",
              "detection_mode": "vehicle"
            }
            """.trimIndent()

        val regions = arrayOf("us", "au")
        val mmc = true
        val direction = true
        val cameraId = ""

        // 4. Loop through images
        val fileResults = mutableListOf<FileResult>()

        imageFiles.forEachIndexed { index, fileName ->
            withContext(Dispatchers.Main) {
                onStatus("Processing ${index + 1}/${imageFiles.size}: $fileName")
            }

            val imgPath = copyAssetToInternalStorage(fileName)

            if (imgPath == null) {
                fileResults.add(FileResult(fileName, "Error: Failed to copy file."))
            } else {
                val procMs =
                    measureTimeMillis {
                        try {
                            val result =
                                Snapshot.engineProc(
                                    imgPath,
                                    configJson,
                                    regions,
                                    mmc,
                                    direction,
                                    cameraId,
                                    Snapshot.currentTimestamp,
                                )
                            fileResults.add(FileResult(fileName, result))
                        } catch (t: Throwable) {
                            Log.e(TAG, "engineProc() error on $fileName", t)
                            fileResults.add(FileResult(fileName, "Error: ${t.message}"))
                        }
                    }
                Log.i(TAG, "Finished $fileName in ${procMs}ms")
            }
        }

        withContext(Dispatchers.Main) { onStatus("Done") }

        return RecognitionOutcome.Success(
            RecognitionInfo("Found ${imageFiles.size} images.", fileResults),
        )
    }

    /**
     * Copy an asset file into internal storage. Runs on the caller's thread.
     * Call from Dispatchers.IO.
     */
    private fun copyAssetToInternalStorage(assetFileName: String): String? =
        try {
            assets.open(assetFileName).use { input ->
                val outFile = File(filesDir, assetFileName)
                // Optional: Check if file exists to skip copying if you want caching
                if (outFile.exists()) return outFile.absolutePath

                FileOutputStream(outFile).use { output ->
                    input.copyTo(output)
                }
                outFile.absolutePath
            }
        } catch (e: IOException) {
            Log.e(TAG, "Failed to copy asset $assetFileName", e)
            null
        }
}

private sealed interface UiState {
    data object Loading : UiState

    data class Success(val info: RecognitionInfo) : UiState

    data class Error(val message: String) : UiState
}

@Composable
private fun plateRecScreen(
    // Status callback is invoked from background work to update the UI text.
    runRecognition: suspend (onStatus: suspend (String) -> Unit) -> RecognitionOutcome,
) {
    var ui by remember { mutableStateOf<UiState>(UiState.Loading) }
    var statusMessage by remember { mutableStateOf("Preparing…") }

    // Launch after first composition; heavy work happens on Dispatchers.IO inside runRecognition()
    LaunchedEffect(Unit) {
        ui = UiState.Loading
        val setStatus: suspend (String) -> Unit = { msg -> statusMessage = msg }

        ui =
            try {
                when (val outcome = runRecognition(setStatus)) {
                    is RecognitionOutcome.Success -> UiState.Success(outcome.info)
                    is RecognitionOutcome.Failure -> UiState.Error(outcome.message)
                }
            } catch (t: Throwable) {
                UiState.Error("Unexpected error: ${t.message}")
            }
    }

    Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
        Box(
            Modifier
                .padding(innerPadding)
                .fillMaxSize(),
        ) {
            when (val s = ui) {
                UiState.Loading -> loadingContent(statusMessage)
                is UiState.Success -> {
                    Column(
                        modifier =
                            Modifier
                                .fillMaxSize()
                                .verticalScroll(rememberScrollState())
                                .padding(16.dp),
                    ) {
                        Text(s.info.summary)
                        Spacer(modifier = Modifier.height(8.dp))
                        s.info.results.forEach { result ->
                            accordionItem(fileName = result.fileName, result = result.result)
                            Spacer(modifier = Modifier.height(8.dp))
                        }
                    }
                }
                is UiState.Error ->
                    Text(
                        text = s.message,
                        modifier =
                            Modifier
                                .fillMaxSize()
                                .padding(16.dp),
                        color = androidx.compose.ui.graphics.Color.Red,
                    )
            }
        }
    }
}

@Composable
private fun accordionItem(
    fileName: String,
    result: String,
) {
    var expanded by remember { mutableStateOf(false) }

    Card(
        modifier = Modifier.fillMaxWidth(),
        onClick = { expanded = !expanded },
    ) {
        Column(
            modifier =
                Modifier
                    .fillMaxWidth()
                    .padding(12.dp),
        ) {
            Text(text = fileName)
            if (expanded) {
                Spacer(modifier = Modifier.height(8.dp))
                Text(text = result)
            }
        }
    }
}

@Composable
private fun loadingContent(message: String) {
    Column(
        modifier =
            Modifier
                .fillMaxSize()
                .padding(24.dp),
        verticalArrangement = Arrangement.Center,
        horizontalAlignment = Alignment.CenterHorizontally,
    ) {
        CircularProgressIndicator()
        Spacer(modifier = Modifier.height(12.dp))
        Text(message)
    }
}

@Preview(showBackground = true)
@Composable
private fun previewPlateRec() {
    platerectestTheme { Text("Preview") }
}
